classdef gaussianOrthoLocalModel
    % GAUSSIANORTHOLOCALMODEL is used for a local model with orthogonal
    % gaussian(s)
    %
    % This class generates a local model object with all relevant
    % information for its orthogonal Gaussians. It delivers all local
    % properties of the local models and the methods to calculate the
    % membership function values and the centers und standard deviations of
    % the Gaussians out of the corners.
    %
    %
    % PROPERTIES:
    %
    % center                    (G x nz)    center of the gaussian
    %                                       (default: []).
    %
    % lowerLeftCorner           (G x nz)    lower left corner of the
    %                                       rectangle surrounding the
    %                                       gaussian (default: []).
    %
    % localLossFunctionValue    (1 x 1)     local loss function value
    %                                       (default: []).
    %
    % standardDeviation         (G x nz)    standard deviations of all
    %                                       gaussians (default: []).
    %
    % upperRightCorner          (G x nz)    upper right corner of the
    %                                       rectangle surrounding the
    %                                       gaussian (default: []).
    %
    % parameter                 (nx x 1)    parameter of the local model
    %                                       (default: []).
    %
    % zLowerBound               (1 x nz)    lower boundary of the training
    %                                       data (default: []).
    %
    % zUpperBound               (1 x nz)    upper boundary of the training
    %                                       data (default: []).
    %
    %
    % EXPLANATIONS:
    %
    %   "standardDeviation":
    %   This property stores the standard deviations of each Gaussians, that
    %   are used to describe the validity area of the local model object.
    %
    %   "lowerLeftCorner":
    %   This property stores the lower left corners of each Gaussian, that
    %   are used to describe the validity area of the local model object.
    %
    %   "upperRightCorner":
    %   This property stores the upper right corners of each Gaussian, that
    %   are used to describe the validity area of the local model object.
    %
    %   "center":
    %   This property stores the centers of each Gaussian, that
    %   are used to describe the validity area of the local model object.
    %
    %   "parameter":
    %   This property contains the parameter vector of the local model,
    %   which is used to calculate the local model output.
    %
    %   "localLossFunctionValue":
    %   This property stores the value of the local loss function for this
    %   local model object. This is needed to find the worst performing
    %   local model.
    %
    %   "zLowerBound":
    %   This property stores the lower boundary of the design space. This
    %   helps to avoid extrapolation in case of recalculating with another
    %   data set (instead the one used for training).
    %
    %   "zUpperBound":
    %   This property stores the upper boundary of the design space. This
    %   helps to avoid extrapolation in case of recalculating with another
    %   data set (instead the one used for training).
    %
    %
    % SYMBOLS AND ABBREVIATIONS
    %
    % LM:  Local model
    %
    % p:   Number of inputs (physical inputs)
    % q:   Number of outputs
    % N:   Number of data samples
    % M:   Number of LMs
    % nx:  Number of regressors (x)
    % nz:  Number of regressors (z)
    % G:   Number of Gaussians
    %
    %
    % LMNtool - Local Model Network Toolbox
    % Tobias Ebert, 18-October-2011
    % Institute of Mechanics & Automatic Control, University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles
    
    
    properties
        center = [];                % (G x nz) center of the gaussian
        lowerLeftCorner = [];       % (G x nz) lower left corner of the rectangle surrounding the gaussian
        localLossFunctionValue = [];% (1 x 1) local loss function value
        standardDeviation = [];     % (G x nz) standard deviations of all gaussians
        upperRightCorner = [];      % (G x nz) upper right corner of the rectangle surrounding the gaussian
        parameter = [];             % (nx x 1) parameter of the local model
        zLowerBound = [];           % (1 x nz) lower boundary of the training data
        zUpperBound = [];           % (1 x nz) upper boundary of the training data
    end
    
    
    methods
        MSFValue = calculateMSF(obj, zRegressor, indexGaussians)
    end
    
    
    methods(Static)
        [center,standardDeviation] = corner2Center(lowerLeftCorner,upperRightCorner,smoothness)
    end
    
    
    methods
        function obj = gaussianOrthoLocalModel(lowerLeftCorner,upperRightCorner,smoothness)
            % constructor of the class
            if exist('lowerLeftCorner','var')
                obj.lowerLeftCorner = lowerLeftCorner;
            end
            if  exist('upperRightCorner','var')
                obj.upperRightCorner = upperRightCorner;
            end
            if exist('lowerLeftCorner','var') && exist('upperRightCorner','var') && exist('smoothness','var')
                [obj.center,obj.standardDeviation] = obj.corner2Center(lowerLeftCorner,upperRightCorner,smoothness);
            end
        end
    end
    
end
