classdef lossFunction
    % LOSSFUNCTION is a class for loss function calculation.
    %
    % Different types of loss function values can be calculated, such as the local, the global and a
    % penalty loss function value. The penalty loss function can be used for approximation of the
    % optimal tradeoff between bias and variance error, based only on the training data and without
    % a separate validation data set and without the application of cross-validation mehtods. This
    % can be used as a first insight for finding the optimal model complexity.
    %
    %
    % PROPERTIES:
    %
    %   lossFunctionGlobal  (string)    Global lossfunction type. Options
    %                                   are: (default:'NRMSE')
    %                                   'MSE'
    %                                   'RMSE' 
    %                                   'NMSE' 
    %                                   'NRMSE'
    %                                   'MISCLASS'
    %                                   'R2'
    %
    %   lossFunctionLocal   (string)    Local lossfunction type. Options
    %                                   are (default: 'RSE'):
    %                                   'SE'
    %                                   'RSE'
    %                                   'DRSE'
    %                                   'MISCLASS'
    %
    %   noiseVariance       (scalar)    User defined process noise variance
    %                                   (optional, default: []).
    %
    %   complexityPenalty   (scalar)    Complextiy penalty factor. Used for penalty loss function
    %                                   calculation (optional, default: 1)
    %
    %   estNoiseVariance    (scalar)    Estimate noise variance. Used for
    %                                   calculating the AIC (default: [])
    %
    %   numberOfCVgroups    (scalar)    Number of groups, the data should be divided in for the cross validation evaluation
    %
    % EXPLENATIONS:
    %
    %   "lossfunctionGlobal":
    %   This property defines which criterion should be used to calculate
    %   the global lossfunction value, i.e. which global loss function
    %   should be evaluated.
    %
    %   "lossfunctionLocal":
    %   This property defines which criterion should be used to calculate
    %   the local lossfunction value, i.e. which local loss function
    %   should be evaluated.
    %
    %   "noiseVariance":
    %   This (optional) property allows the user to define a certain process
    %   noise variance, based on prior knowledge about the process noise.
    %
    %   "complexityPenalty":
    %   Factor that penalizes the number of overall model parameters. The penalty loss is defined as
    %   GPE = logLikelihood/N  +  complexityPenalty * numberOfAllParameters/N  , i.e.
    %       = bias term        +  complexityPenalty * variance term
    %
    %   "estNoiseVariance":
    %   Calculation of the Generalized Prediction Error (GPE), see
    %   [John Moody: "Prediction Risk and Architecture Selection for Neural Network", 1994].
    %   The noise variance must be estimated with a given LMN, otherwise it is roughly approximated
    %   with ( noise variance = MSE of current LMN ), which results in the AIC for unknown noise variance.
    %
    %
    % SYMBOLS AND ABBREVIATIONS
    %
    % LM:  Local model
    %
    % p:   Number of inputs (physical inputs)
    % q:   Number of outputs
    % N:   Number of data samples
    % M:   Number of LMs
    % nx:  Number of regressors (x)
    % nz:  Number of regressors (z)
    %
    %
    % LMNtool - Local Model Network Toolbox
    % Benjamin Hartmann, 04-April-2012
    % Institute of Mechanics & Automatic Control, University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles
    
    % 16.11.11 help updated (TE) 
    % 13.12.11 Property lossFunctionPenalty removed and update for
    %          calcPenaltyLossFunction, B.Hartmann
    % 15.12.11 Added properties 'noiseVariance' and 'complexityPenalty'. Help updated, B.Hartmann
    % 16.03.12 Added properties 'crossvalidationValues'; Added static method 'crossvalidation' (JB)
    % 15.08.12 Removed property 'crossvalidationValues; Method 'crossvalidation' no longer a static method (JB)
    % 16.08.12 Added property: kFold (JB)

    
    properties
        lossFunctionGlobal = 'NRMSE'; % Global lossfunction type ('MSE','RMSE','NMSE','NRMSE', 'MISCLASS' and 'R2' can be chosen).
        lossFunctionLocal  = 'RSE';   % Local lossfunction type ('SE','RSE','DRSE' and 'MISCLASS' can be chosen).
        noiseVariance      = [];      % This property allows the user to define a certain process noise variance, based on prior knowledge about the process noise.
        complexityPenalty  = 1;       % Complexity penalty.
        estNoiseVariance   = inf;     % 
        numberOfCVgroups    = [];     % Number of groups, the data should be divided in for the cross validation evaluation
    end
    
    methods
        [J] = calcGlobalLossFunction(obj, output, outputModel, dataWeighting, outputWeighting, criterion)                         % Calculate the global loss function of the global model object.
        [J] = calcLocalLossFunction(obj, output, outputModel, validityFunctionValue, dataWeighting, outputWeighting, criterion)   % Calculate the local loss function for each local model object.
        [J nPara CVScore nEff AIC] = calcPenaltyLossFunction(obj)   
        [J JAll]               = calcLocalCVLossFunction(obj, LM)   
        [nEff]                 = calcNEff(obj)
        [nEffi]                = calcNEffi(obj)
        [CVScore nEff LOOE nEffAllLM]    = calcLOOError(obj)
        [errorbar df1 df2 estVar]  = calcErrorbar(obj, inputnew)
        [crossValidationValues,obj] = crossvalidation(obj,numberOfBatches,modelComplexities,ini);
    end

end

