classdef modelHistory
    % MODELHISTORY creates a history object containing all relevant
    % information about every iteration step.
    %
    % This super-class adds the ability to the sub-classes to store all
    % necessary properties in terms of history. Therefore it updates in 
    % each iteration its properties by the loss function values, the split
    % characteristics and the leaf models, i.e. in each iteration the size
    % of the properties changes.
    %
    %
    % PROPERTIES:
    % 
    % currentNumberOfParameters: (1 x iter) Current number of
    %                                       parameters (default: []).
    %
    % currentNumberOfLMs: (1 x iter)    Current number of local models
    %                                   (default: []).
    %
    % globalLossFunction: (1 x iter)    Loss function values for the
    %                                   overall model (default: []).
    %
    % penaltyLossFunction: (1 x iter)   Complexity penalized loss function
    %                                   values (default: []).
    %
    % splitLM: (iter-1 x 1)   History of LMs which are split (default: []).
    %
    % splitDimension: (iter-1 x 1)  History of dimensions of the split
    %                               (default: []).
    %
    % splitRatio: (iter-1 x 1)  History of ratio of the split
    %                           (default: []).
    %
    % leafModelIter: (1 x iter) History of the active models per iteration
    %                           (default: cell(1x1)).
    %
    % trainingTime: (1 x iter)  Overall Training time up to this iteration
    %
    % displayMode: (logical)    Flag to de-/activate printouts to the
    %                           command window during training(default:
    %                           true).
    %
    % iteration: (1 x 1)    Current number of iterations (default: zero).
    %
    % kFoldCVlossFunction (1 x iter) k-fold cross validation RMSE-value
    %
    %
    % SYMBOLS AND ABBREVIATIONS:
    %
    % iter:  number of iteration during the training
    %
    % EXPLANATIONS:
    %
    %   "currentNumberOfParameters": This property stores the current
    %   number of parameters for each iteration, so that a good idea of the
    %   model complexity is given.
    %
    %   "currentNumberOfLMs": This property stores the current number of
    %   local models for each iteration, so that a good idea of the model
    %   complexity is given.
    %
    %   "globalLossFunction": This property stores for each iteration the
    %   global loss function value, so it is possible to review the
    %   progression of the global error.
    %
    %   "penaltyLossFunction": This property stores for each iteration the
    %   penalty loss function value, so it is possible to review the
    %   progression of the penalty error.
    %
    %   "splitLM": This property stores for each iteration the local model
    %   object, which was splitted.
    %
    %   "splitDimension": This property stores for each iteration the
    %   dimension of the split done in the premises space.
    %
    %   "splitRatio": This property stores for each iteration the
    %   ratio of the split done in the premises space.
    %
    %   "leafModelIter": This property stores in each iteration the index
    %   vector that specifies the leaf models, so its possible to use a net
    %   with lower complexity if the penalty error is too high.
    %
    %   "displayMode": This property is a flag that activate und deactivate
    %   printouts to the command window during the training. If the user is
    %   not interested in such information, printouts should be deactivated
    %   to have a better performance.
    %
    %   "iteration": This property stores the current iteration during the
    %   training, so that every other property of the MODELHISTORY class
    %   can be related to it.
    %
    
    % LMNtool - Local Model Network Toolbox
    % Torsten Fischer, 18-November-2011
    % Institute of Mechanics & Automatic Control, University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles
        
    % Changelog:
    % - Added "currentNumberOfParameters", 19-December-2011, B. Hartmann
    % - Added "kFoldCVlossFunction", 16-August-2012, B. Julian
    % - Added "akaikeWeights" as a dependent property and the static method
    % "calcAkaikeWeights", 29-Nov-2012, B. Julian
    
    properties
        currentNumberOfParameters = [];    % (1 x iter)      Overall number of model parameters for each iteration
        currentNumberOfEffParameters = []; % (1 x iter)      Effective number of model parameters for each iteration
        currentNumberOfLMs  = [];          % (1 x iter)      Overall number of models for each iteration
        globalLossFunction  = [];          % (1 x iter)      Loss function values for the overall model
        penaltyLossFunction = [];          % (1 x iter)      Complexity penalized loss function values
        AICc                = [];          % (1 x iter)      Akaike Information Criterion (AIC) for each iteration
        CVLossFunction      = [];          % (1 x iter)      Leave-one-out cross validation scores
        splitLM             = [];          % (iter-1 x 1)    History of LMs which are split
        splitDimension      = [];          % (iter-1 x 1)    History of dimensions of the split
        splitRatio          = [];          % (iter-1 x 1)    History of ratio of the split
        leafModelIter       = cell(1,1);   % (1 x iter)      History of the active models per iteration
        trainingTime        = [];          % (1 x iter)      Overall Training time up to this iteration
        displayMode         = true;        % (logical)       Flag to de-/activate printouts to the command window during training
        iteration           = zeros(1,1);  % (1 x 1)         Stores the number of the used iteration
        validationDataLossFunction  = [];  % (1 x iter)      Validation data loss function values for the overall model
        testDataLossFunction        = [];  % (1 x iter)      Test data loss function values for the overall model
        kFoldCVlossFunction = [];          % (1 x iter)      k-fold cross validation RMSE-value
    end
    
    properties (SetAccess = private, GetAccess = public)
        akaikeWeights       = [];          % (1 x iter)      Akaike weight for each iteration/model (with different complexity).
    end
    
    methods
        obj = writeHistory(obj,LMobj,tStart)
    end
    
    methods (Static=true)
        akaikeWeights = calcAkaikeWeights(aicValues);
    end
    
    methods
        
        % Set and get methods
        
        % Get method for the akaike weights.
        function aW = get.akaikeWeights(obj)
            % The Akaike weights can only be calculated, if the property
            % AICc is not empty
            if isempty(obj.AICc)
                aW = [];
                fprintf('\n Because there are no AICc values, no Akaike weights can be calculated.\n \n');
            else
                aW = obj.calcAkaikeWeights(obj.AICc);
            end
            
        end
        
    end
    
end

