classdef terminationCriterions
    % TERMINATIONCRITERIONS incldes all common termination criterions for
    % local model algorithms
    %
    % PROPERTIES:
    %
    %   maxNumberOfLM:
    %   It is a constant describing the maximum number of local models
    %   during the training. Here linear local models. It is a abort
    %   criterion.
    %
    %   minError:
    %   It is a constant decribing the minimum error to be reached during
    %   the training. It is a abort criterion.
    %
    %   maxNumberOfParameters:
    %   It is a constant describing the maximum number of parameter during
    %   the training. It is a abort criterion.
    %
    %   maxTrainTime:
    %   This scalar defines the time in minutes that the algorithm is
    %   allowed to use for training. It is an abort criterion.
    %
    %   maxIterations:
    %   This scalar defines the maximum number of iteration, that polymot
    %   is allowed to calculate during the training. It is a abort
    %   criterion.
    %
    %   maxPenaltyDeterioration:
    %   This scalar defines the maximum number of deteriorations of the
    %   penalty loss function opver its developing. It is an abort
    %   criterion.
    %
    %   complexityPenalty:
    %   This is a 'fuddle'-parameter, who desrcibes the penalty factor for
    %   a higher complexity (to avoid overfitting).
    %
    % SYMBOLS AND ABBREVIATIONS:
    %
    % N:    Number of samples.
    % q:    Number of outputs.
    % nx:   Number of rule consequents.
    % nz:   Number of rule premises.
    % M:    Number of local models.
    
    % LMNTool - Nonlinear System Identification Toolbox
    % Tobias Ebert, 12-September-2012
    % Institute of Mechanics & Automatic Control
    % University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles et. al.
    
    properties
        maxNumberOfLM = inf;            % 1 x 1 Maximum number of LLMs termination criterion
        minError = 0;                   % 1 x 1 Loss function threshold termination criterion
        maxNumberOfParameters = inf;    % 1 x 1 Maximum number of Parameter termination criterion
        maxTrainTime = inf;             % 1 x 1 Maximum allowed time for the calculation
        maxIterations = inf;            % 1 x 1 Maximum number of iterations termination criterion
        maxPenaltyDeterioration = 2;	% 1 x 1 Maximum number of deterioration of the penalty loss function
        maxValidationDeterioration = 2;
        minPerformanceImprovement = 1e-12; % 1 x 1 Minimum performance improvement of termination criterion
    end
    
    methods
        result = checkTerminationCriterions(LMNobj)
    end
    
    methods
        % SET and GET methods
        function obj = set.maxNumberOfLM(obj,value)
            % Check if the max number of LM is greater than one
            if value >= 1
                % Check if the value is a whole number
                if value - floor(value) > eps
                    warning('terminationCriterions:set:maxNumberOfLM','max number of local models must be a whole number! The given value is rounded down.')
                end
                obj.maxNumberOfLM = floor(value);
            else
                error('terminationCriterions:set:maxNumberOfLM','max number of local models must be at least one!')
            end
        end
        
        function obj = set.maxNumberOfParameters(obj,value)
            % Check if the max number of parameters is greater than zero
            if value > 0
                obj.maxNumberOfParameters = value;
            else
                error('terminationCriterions:set:maxNumberOfParameters','max number of parameters must be greater than zero!')
            end
        end
        
        function obj = set.maxTrainTime(obj,value)
            % Check if the max training time is greater than zero
            if value > 0
                obj.maxTrainTime = value;
            else
                error('terminationCriterions:set:maxTrainTime','max training time must be greater than zero!')
            end
        end
        
        function obj = set.minError(obj,value)
            % Check if the min error is greater than zero
            if value >= 0
                obj.minError = value;
            else
                error('terminationCriterions:set:minError','min training error must be at least zero!')
            end
        end
        
        function obj = set.maxIterations(obj,value)
            % Check if the max number of iterations is greater than one
            if value >= 1
                % Check if the value is a whole number
                if value - floor(value) > eps
                    warning('terminationCriterions:set:maxIterations','max number of iterations must be a whole number! The given value is rounded down.')
                end
                obj.maxIterations = floor(value);
            else
                error('terminationCriterions:set:maxIterations','max number of iterations must be at least one!')
            end
        end
        
        function obj = set.maxPenaltyDeterioration(obj,value)
            % Check if the max penalty deterioration is greater than one
            if value >= 1
                % Check if the value is a whole number
                if value - floor(value) > eps
                    warning('terminationCriterions:set:maxPenaltyDeterioration','max penalty deterioration must be a whole number! The given value is rounded down.')
                end
                obj.maxPenaltyDeterioration = floor(value);
            else
                error('terminationCriterions:set:maxPenaltyDeterioration','max penalty deterioration must be at least one!')
            end
        end
        
        function obj = set.maxValidationDeterioration(obj,value)
            % Check if the max validation deterioration is greater than one
            if value >= 1
                % Check if the value is a whole number
                if value - floor(value) > eps
                    warning('terminationCriterions:set:maxValidationDeterioration','max validation deterioration must be a whole number! The given value is rounded down.')
                end
                obj.maxValidationDeterioration = floor(value);
            else
                error('terminationCriterions:set:maxValidationDeterioration','max validation deterioration must be at least one!')
            end
        end
        
        function obj = set.minPerformanceImprovement(obj,value)
            % Check if the min performance improvement is greater than zero
            if value >= 0
                obj.minPerformanceImprovement = value;
            else
                error('terminationCriterions:set:minError','min training error must be at least zero!')
            end
        end
        
    end
    
end

