function [x,y] = getCriterion(modelObject,criterion,modelComplexities)
% Function calculates loss-function values for all modelcomplexities
%
% [x,y] = getCriterion(modelObject,criterion)
%
% Inputs
%
% modelObject:          model  object
% criterion:            String to decide which criterion should be
%                       evaluated
% modelComplexities:    (1 x number of model complexities) vector
%
% Outputs
%
% x:                    (1 x number of model complexities) vector
% y:                    (1 x number of model complexities) vector

% Julian Belz, 19-Mrz-2012
% Institut fr Mess- und Regelungstechnik, Universitt Siegen, Deutschland
% Institute of Measurement and Control, University of Siegen, Germany
% Copyright (c) 2012 by Julian Belz

if nargin < 3
    modelComplexities = 1:size(modelObject.history.leafModelIter,2);
end


x   = modelComplexities;
y   = zeros(size(x));
if strcmp(criterion,'AIC')
    % AIC
    for jj = [x; 1:size(x,2)]
        
        modelObject.leafModels = modelObject.history.leafModelIter{jj(1)};
        modelObject.outputModel = modelObject.calculateModelOutput(modelObject.unscaledInput, modelObject.unscaledOutput);
        y(jj(2)) = modelObject.calcPenaltyLossFunction;
        
    end
else
    % adjusted R^2 or R^2
    for jj = [x;1:size(x,2)]
        
        modelObject.leafModels = modelObject.history.leafModelIter{jj(1)};
        yModel = modelObject.calculateModelOutput(modelObject.unscaledInput, modelObject.unscaledOutput);
        
        R2 = modelObject.calcGlobalLossFunction(modelObject.output,...
            yModel,modelObject.dataWeighting,modelObject.outputWeighting,'R2');
        
        if strcmp(criterion,'R2')
            y(jj(2)) = R2;
        else
            % Determing the number of parameters
            allParameters   = cell2mat({modelObject.localModels(modelObject.leafModels).parameter});
            noParameters    = sum(sum(allParameters~=0));
            noSamples = size(modelObject.input,1);
            
            % adjusted R^2
            y(jj(2)) = R2-(1-R2)*noParameters/(noSamples-noParameters-1);
        end
        
    end
end

end