classdef hilomot < sigmoidGlobalModel & dataSet
    % HILOMOT creates a HIerarchical LOcal MOdel Tree object with the HILOMOT algorithm.
    %
    % The output of a local model network is calculated as the weighted sum of local submodels that
    % are of polynomial type, i.e. linearly parametrized. By default, the HILOMOT algorithm uses
    % local 'affine' models, but extensions to higher order polynomials are straightforward.
    %
    % The local model weighting is performed by the validity functions that are determined
    % heuristically with a incrementally growing tree-search algorithm. In each iteration one local
    % model is added to the global model. HILOMOT models are organized as so-called hinging
    % hyperplane trees. The hinges, i.e. the validity function parameters are optimized with respect
    % to the global model error.
    %
    %
    % 1) MODEL PROPERTIES
    %
    % suggestedNet:               (1 x 1)   Suggested net with best performance/complexity trade-off
    %                                       (default: []).
    % oblique:                    logical   0: axes-orthogonal partitioning, 1: axis-oblique
    %                                       partitioning (default: 1).
    % numberOfPoints:             (1 x 1)   Minimum number of points that have to be in a
    %                                       LM before it is allowed for splitting
    %                                       (default: 0).
    % optGrad:                    logical   0: Numerical Gradient, 1: Analytical Gradient
    %                                       (default: 0).
    % optLOOCV                    logical   If true, split opt. w.r.t. LOO cross val. error
    %
    %
    % 2) DATA PROPERTIES
    %
    % outputModel:                (N x q)   Model outputs.
    % xRegressor:                 (N x nx)  Regression matrix/Regressors for rule consequents.
    % zRegressor:                 (N x nz)  Regression matrix/Regressors for rule premises.
    % phi:                        (N x M)   Validity function matrix.
    %
    %
    %
    %
    % SYMBOLS AND ABBREVIATIONS
    %
    % LM:  Local model
    %
    % p:   Number of inputs (physical inputs)
    % q:   Number of outputs
    % N:   Number of data samples
    % M:   Number of LMs
    % nx:  Number of regressors (x)
    % nz:  Number of regressors (z)
    %
    %
    % HiLoMoT - Nonlinear System Identification Toolbox
    % Benjamin Hartmann, 04-April-2012
    % Institute of Mechanics & Automatic Control, University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles
    
    
    properties
        
        % Model properties
        oblique               = true;  % logical   0: axes-orthogonal partitioning, 1: axis-oblique partitioning
        optGrad               = false; % logical   0: numerical gradient, 1: analytical gradient
        optLOOCV              = false; % logical   If true, split opt. w.r.t. LOO cross val. error
        
        % Data properties
        outputModel           = [];   % (N x q)   Model outputs
        xRegressor            = [];   % (N x nx)  Regression matrix/Regressors for rule consequents
        zRegressor            = [];   % (N x nz)  Regression matrix/Regressors for rule premises
        phi                   = [];   % (N x M)   Validity function matrix
        relationZtoX          = [];   % (1 x n)   Indicates which Regressor z- and x-space have incommon
        
    end
    
    properties(Hidden=true)
        
        % Hidden properties
        numberOfLMReliable         = [];    % (1 x 1)   Number of LMs for a reliable model estimation        
        inputSensitivity           = [];    % (nz x 1)  Sensitivity analysis of zRegressor-inputs. Value 1 has highest importance, value 0 seems to be a redundant input.
        
    end
    
    methods
        
        obj     = train(obj,displayMode)
        obj     = estimateFirstLM(obj)
        [obj, forbiddenSplit] = LMSplit(obj, worstLM, splitDimension)
        [obj, forbiddenSplit] = LMSplitOpt(obj, wIni, worstLM)
        worstLM = findWorstLM(obj)
        
    end
    
    methods(Static)
        
        [y name] = exampleFunctions(u,bsp)
        [J, gradient, parameter, center, phi,  outputModel, pseudoInv] = ...
            obliqueGlobalLossFunction(wRed, w0, xRegressor, zRegressor, output, outputModel,...
            weightedOutputWorstLM, phiWorstLM, smoothness, optGrad, dataWeighting, LOOCV, optLOOCV, SiiAll, ...
            kStepPrediction, xInputDelay, zInputDelay, xOutputDelay, zOutputDelay,...
            leafModels, localModels, worstLM)
        
    end
    
    methods
        %% Constructor
        function obj = hilomot
            if obj.zRegressorDegree == 1
                obj.smoothness = 1.0;
            end
        end
    end
    
end

