classdef hilomotDoE
    % HILOMOTDOE performs model-based experiment design using local model networks.
    %
    %
    % 1) ALGORITHM PROPERTIES & DATA INTERFACE
    %
    %     LMNObject:              object   Local model network object.
    %     candidatePoints:        (? x p)  Candidate points for DoE procedure.
    %     inputMeasured:          (N x p)  Input values from measurement.
    %     inputTarget:            (N x p)  Designed input target values.
    %     output:                 (N x 1)  Measured output values.
    %
    %
    % 2) USER OPTIONS FOR DOE
    %
    %     query:                  string   Choose query strategy for DoE (default: 'worstLM').
    %     variableIncrements:     struct   Delivers min, max and increment value for each input
    %                                      dimension. If this variable is left empty, only candidatePoints are taken
    %                                      into account for query and no next increment is searched (default).
    %     smoothnessQuery:        (1 x 1)  Overlap between the validity functions for query evaluation.
    %     numberOfInitialPoints:  (1 x 1)  Number of points that initialize the HilomotDoE algorithm.
    %     numberOfAllSamples:     (1 x 1)  Maximum number of data points that have to be generated with DoE.
    %     designProgram:          (1 x 1)  1: initialization phase, 2: active learning phase, 3: measurement failure.
    %     flagSpaceFilling:       (1 x 1)  Flag, if last measurement failed (NaN) or if #valid < #initial points.
    %
    %
    % 3) MODELING PROPERTIES
    %
    %     maxNumberOfLM           (1 x 1)  Set maximum number of local models for training.
    %     oblique                 (1 x 1)  0: axes-orthogonal, 1: axes-oblique partitioning (default: 1).
    %     numberOfPoints          (1 x 1)  Minimum number of points in one local model.
    %     smoothnessSplit         (1 x 1)  Overlap between the validity functions for model training.
    %
    %
    % SYMBOLS AND ABBREVIATIONS
    %
    % LM:  Local model
    % p:   Number of inputs (physical inputs)
    % N:   Current number of data samples
    %
    %
    % HiLoMoTDoE - Nonlinear System Identification Toolbox for Design of Experiments
    % Benjamin Hartmann, 05-April-2012
    % Institute of Mechanics & Automatic Control, University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles

    
    properties
        
        % Algorithm properties & data interface
        LMNObject                   % Local model network object.
        candidatePoints             % Candidate points for DoE procedure.
        inputMeasured               % Input values from measurement.
        inputTarget                 % Designed input target values.
        output                      % Measured output values.
        
        % User options for DoE
        query = 'worstLM';          % Choose query strategy for DoE (default: 'worstLM').
        variableIncrements = [];    % Delivers min, max and increment value for each input dimension.
        smoothnessQuery = 1.25;     % Overlap between the validity functions for query evaluation.
        numberOfInitialPoints = []  % Number of points that initialize the HilomotDoE algorithm.
        numberOfAllSamples          % Maximum number of data points that have to be generated with DoE.
        designProgram               % 1: initialization phase, 2: active learning phase, 3: measurement failure.
        flagSpaceFilling = true;    % Flag, if last measurement failed (NaN) or if #valid < #initial points.
        
        % Modeling properties
        maxNumberOfLM               % Set maximum number of local models for training.
        oblique = 1;                % 0: axes-orthogonal, 1: axes-oblique partitioning (default).
        numberOfPoints = 0;         % Minimum number of points in one local model.
        smoothnessSplit = 1.25;     % Overlap between the validity functions for model training.
    end
    
    properties(Hidden=true)
        
        % Hidden modeling properties
        numberOfInputs              % Set number of inputs for training and DoE.
        optGrad = 0;                % 0: numerical gradient (default), 1: analytical gradient.
        complexityPenalty = 1;      % Penalizes too flexible models (default: 2).

    end
    
    methods
        
        obj   = DoEUpdate(obj)
        query = DoEQuery(obj)
        [query dNN] = DoEGenerateNewData(obj,worstLM)
        
    end
    
    methods(Static)
        
        [dataOut MaxVal MinVal] = DoEScaleData(dataIn,MaxVal,MinVal,direction)
        [newPoint dNN] = DoEPMCPoint(numberOfNewPoints, existingPoints, candidatePoints)
        [queryNew dNN] = DoEFindNextIncrement(query, varInc)
        
    end
    
end


