function plotPartitionColored(obj,dimToPlot)
% PLOTPARTITIONCOLORED plots the partitioning of the premise input space of
% the neuro-fuzzy model (1-, 2-, or 3-dimensional plot only).
%
%   plotPartition(obj,dimToPlot)
%
%
% INPUTS:
%
% obj           (object)    local model object containing all relevant net
%                           and data set information and variables
%
% dimToPlot     (optional)  vector of dimensions you want to plot
%
%
% SYMBOLS AND ABBREVIATIONS:
%
% LM:  Local model
%
% p:   Number of inputs (physical inputs)
% q:   Number of outputs
% N:   Number of data samples
% M:   Number of LMs
% nx:  Number of regressors (x)
% nz:  Number of regressors (z)
%
%
% LMNtool - Local Model Network Toolbox
% Torsten Fischer, 15-November-2011
% Institute of Mechanics & Automatic Control, University of Siegen, Germany
% Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles


% get the dimension of the physical input space
dimension = obj.numberOfInputs;

% check which dimension should be plotted
if ~exist('dimToPlot','var')
    if dimension <= 3
        dimToPlot = 1:dimension;
    else
        warning('gaussianOrthoGlobalModel:plotPartition','No dimensions to plot where given')
    end
end

% check if prohibited dimensions for plotting are given
if ~isempty(dimToPlot) && any(dimToPlot > dimension)
    % test for wrong dimensions
    error(['Dimension(s) possible: ' num2str(1:dimension) ', Dimension(s) you want to plot: ' num2str(dimToPlot)])
end

% Set random number generation to a fix start
rng(1)
CMR = rand(numel(obj.leafModels),1); % one fix color for each local model
rng(2)
CMG = rand(numel(obj.leafModels),1); % one fix color for each local model
rng(3)
CMB = rand(numel(obj.leafModels),1); % one fix color for each local model
CM = [CMR CMG CMB];

% start plot
if length(dimToPlot)==1
    plotPartitionColored1D(obj,dimToPlot,CM);
elseif length(dimToPlot)==2
    plotPartitionColored2D(obj,dimToPlot,CM);
else
    plotPartitionColored3D(obj,dimToPlot,CM);
end

end

%% 1D
function [hmodel,hdata] = plotPartitionColored1D(obj,dimToPlot,CM)

% check all local model obejcts
for LM = find(obj.leafModels)
    
    % initialize the complete surface matrix
    X = []; Y = [];
    
    % get the surfaces of all Gaussians of one local model object
    for Gaussians = 1:size(obj.localModels(LM).center,1)
        X_neu = [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]';
        Y_neu = [0 0 1 1]';
        
        % superposition all surfaces of one local model object
        X = [X X_neu];
        Y = [Y Y_neu];
    end
    hold on
    % do patches for the current leaf model
    patch(X,Y,CM(LM,:));
end
end

function [hmodel,hdata] = plotPartitionColored2D(obj,dimToPlot,CM)

% check all local model obejcts
for LM = find(obj.leafModels)
    
    % initialize the complete surface matrix
    X = []; Y = [];
    
    % get the surfaces of all Gaussians of one local model object
    for Gaussians = 1:size(obj.localModels(LM).center,1)
        X_neu = [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]';
        Y_neu = [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2))]';
        
        % superposition all surfaces of one local model object
        X = [X X_neu];
        Y = [Y Y_neu];
    end
    hold on
    % do patches for the current leaf model
    patch(X,Y,CM(LM,:));
end
end

function [hmodel,hdata] = plotPartitionColored3D(obj,dimToPlot,CM)

% check all local model obejcts
for LM = find(obj.leafModels)
    
    % initialize the complete surface matrix
    X = []; Y = []; Z = [];
    
    % get the surfaces of all Gaussians of one local model object
    for Gaussians = 1:size(obj.localModels(LM).center,1)
        
        X_neu =[[obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]' ...
            [obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1))]'  ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(1)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(1))]'];
        Y_neu =[[obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2))]'  ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2))]' ...
            [obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(2))]'];
        Z_neu =[[obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3))]'  ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3))]' ...
            [obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).lowerLeftCorner(Gaussians,dimToPlot(3))]' ...
            [obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3)) obj.localModels(LM).upperRightCorner(Gaussians,dimToPlot(3))]'];
        
        % superposition all surfaces of one local model object
        X = [X X_neu];
        Y = [Y Y_neu];
        Z = [Z Z_neu];
    end
    hold on
    % do patches for the current leaf model
    patch(X,Y,Z,CM(LM,:));
end
view(3)
end