function plotHandles = visualizeSeperateLocalModels(obj,input,axesHandle)
% This function visualizes the active local models on their own and is only
% suited for one or two dimensional input spaces.
%
% visualizeSeperateLocalModels(obj,input,axesHandle)
%
% INPUTS
%
% obj:      Trained local model network
%
% input:    Data, on which the output of the local models are calculated.
%           If no or an empty 'input' is passed, the training data of the
%           local model network is utilized for visualization.
%
% axesHandle: Handle to axis object, where the local models should be
%             plotted in.
%
% 



% obj = lmn;

if ~exist('axesHandle','var') || isempty(axesHandle)
    axesHandle = gca;
end

if ~exist('input','var') || isempty(input)
    input = obj.input;
end

if size(input,2) > 2
    fprintf('This function is suited only for one or two dimensional input spaces.\n \n');
    return;
end

if size(obj.output,2) > 1
    fprintf('This function is only suited for models with single output.\n \n');
    return;
end

%
hold(axesHandle,'on');

% Determine active models
leafModels = find(obj.leafModels);

% Predefine matrix, where every column will contain the output of one
% active local model (this function is not suited for models with multiple
% outputs)
yHats = zeros(size(input,1),size(leafModels,2));

% Create x-regressor
X = obj.data2xRegressor(input);

% Create z-regressor
Z = obj.data2zRegressor(input);

% Calculate validity values for the given inputs and the corresponding
% active local model
phi     = obj.calculatePhi(Z,obj.leafModels);
if iscell(phi)
    phi = cell2mat(phi);
end

% Reshape the input matrix for the surf plot, if the input space
% dimensionality equals two
if size(input,2) == 2
    
    % Determine the number of points per axis
    numberOfPointsPerAxis = floor(sqrt(size(input,1)));
    
    u1g = reshape(input(1:numberOfPointsPerAxis^2,1),numberOfPointsPerAxis,numberOfPointsPerAxis);
    u2g = reshape(input(1:numberOfPointsPerAxis^2,2),numberOfPointsPerAxis,numberOfPointsPerAxis);
end

% Initialize cell array, that contains all plot handles
plotHandles = cell(1,size(leafModels,2));

% Plot every active, local model
for ii=[leafModels;1:size(leafModels,2)]
    
    % Calculate output of active local model
    yHat = X*obj.localModels(1,ii(1)).parameter;
    
    % Set all outputs of the active local model with a validity value of
    % less than 0.5 to NaN
    idxNaN = phi(:,ii(2)) < 0.5;
    yHat(idxNaN,:) = NaN;
    
    % Save result to the pre-defined matrix
    yHats(:,ii(2)) = yHat;
    
    % Reshape output vector for the surf plot, if input space
    % dimensionality equals two
    if size(input,2) == 2
        yHg = reshape(yHat(1:numberOfPointsPerAxis^2,1),numberOfPointsPerAxis,numberOfPointsPerAxis);
    end
    
    if size(input,2) == 2
        plotHandles{1,ii(2)} = surf(axesHandle,u1g,u2g,yHg);
    elseif size(input,2) == 1
        plotHandles{1,ii(2)} = plot(axesHandle,input,yHat,'LineWidth',2);
        set(plotHandles{1,ii(2)},'Color',[1 0 0]);
    else
        fprintf('This function is not able to visualize the local models for more than two inputs.\n \n');
        return;
    end
    
end

% hSurfModel = surf(axesHandle,

set(axesHandle,'FontName','Arial','FontSize',14);
grid(axesHandle,'on');

end