function [center,standardDeviation] = corner2Center(lowerLeftCorner,upperRightCorner,smoothness)
% CORNER2CENTER convert the corners of the validity areas to centers and
% standard deviations of the resulting membership functions (Gaussians).
%
% [center, standardDeviation] = corner2Center(lowerLeftCorner, upperRightCorner, smoothness)
%
%
% OUTPUTS:
%
%   center:             (G x nz)    centers of the Gaussian
%
%   standardDeviation   (G x nz)    standard deviations of the Gaussians
%
%
% INPUTS:
%
%   lowerLeftCorner:    (G x nz)	lower left corners of the hyper-rectangles
%
%   upperRightCorner:   (G x nz)	upper right corners of the hyper-rectangles
%
%   smoothness:         (1 x 1)     smoothness factor
%
%
% SYMBOLS AND ABBREVIATIONS:
%
% LM:  Local model
%
% G:   Number of Gaussians
% p:   Number of inputs (physical inputs)
% q:   Number of outputs
% N:   Number of data samples
% M:   Number of LMs
% nx:  Number of regressors (x)
% nz:  Number of regressors (z)
%
%
% LMNtool - Local Model Network Toolbox
% Torsten Fischer, 16-November-2011
% Institute of Mechanics & Automatic Control, University of Siegen, Germany
% Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles

% 18.11.11  help updated

% initialize empty output parameters 
center = [];
standardDeviation = [];

% check if the number of corners is correct
if isempty(lowerLeftCorner)
    disp(['number of lower left corners: ' num2str(size(lowerLeftCorner,1))])
    disp(['number of upper right corners: ' num2str(size(upperRightCorner,1))])
    warning('localModelGaussianOrtho:corner2center','No lower left corners are given. The center and standard deviation are empty!')
    return
elseif isempty(upperRightCorner)
    disp(['number of lower left corners: ' num2str(size(lowerLeftCorner,1))])
    disp(['number of upper right corners: ' num2str(size(upperRightCorner,1))])
    warning('localModelGaussianOrtho:corner2center','No upper right corners are given. The center and standard deviation are empty!')
    return
elseif numel(lowerLeftCorner)~=numel(upperRightCorner)
    disp(['number of lower left corners: ' num2str(size(lowerLeftCorner,1))])
    disp(['number of upper right corners: ' num2str(size(upperRightCorner,1))])
    warning('localModelGaussianOrtho:corner2center','Number of lower left and upper right corners must be the same. The center and standard deviation are empty!')
    return
elseif ~isscalar(smoothness)
    error('localModelGaussianOrtho:corner2center','Smoothness must be a scalar!')
end

% initialize output paramters with zeros
center = zeros(size(lowerLeftCorner));
standardDeviation = zeros(size(lowerLeftCorner));

% Set smoothness to default if not defined
if ~exist('smoothness','var')
    smoothness = 1;
end

% Evaluate centers and standard deviations for the given corners
for Gaussian = 1:size(center,1) % for-loop over all Gaussians
    % Center of a hyper-rectangle
    center(Gaussian,:) = (upperRightCorner(Gaussian,:) + lowerLeftCorner(Gaussian,:)) / 2;
    % Standard deviation proportional to the size of the hyper-rectangle, proportionality factor: 2.5
    standardDeviation(Gaussian,:) = smoothness * (upperRightCorner(Gaussian,:) - lowerLeftCorner(Gaussian,:)) / 2.5;
    
end

