classdef lolimot < gaussianOrthoGlobalModel & dataSet 
    % The LOLIMOT class generates a net object consisting of local models.
    %
    % Each model has a certain area, where it is valid. This area is
    % specified by a validity function. In case of a LOLIMOT object,
    % Gaussian are used to decribe the local areas. To get a 'Partition
    % of Unity' the Gaussians are normed. The training  is pretty fast,
    % because the algorithmn split only orthogonal halfs. Therefore it is
    % not very flexible. The local models themselves are linear, i.e.
    % hyperplanes. This keeps the number of parameter very handly even for
    % higher dimensional problems.
    %
    %
    % PROPERTIES:
    %
    % numberOfLMReliable:  (1 x 1)  Iteration that guarantee a reliable
    %                               model (default: 0).
    %
    % suggestedNet:  (1 x 1)    Iteration with the best performance/complexity
    %                           trade-off (default: 0).
    %
    % idxAllowedLM:  (1 x M)    Stores the local models that are allowed to
    %                           be splitted (default: []).
    %
    %
    % splits:  (1 x 1)  Number of splits in each dimension during the
    %                   training (default: 1).
    %
    % outputModel:  (N x q)     Global model output (default: []).
    %
    % xRegressor:  (N x nx)     Regression matrix/Regressors for rule
    %                           consequents (default: []).
    %
    % zRegressor:  (N x nz)     Regression matrix/Regressors for rule
    %                           premises (default: []).
    %
    % MSFValue:  (1 x M)    Membership function values for each local
    %                       model (default: cell(1,1)).
    %
    %
    % SYMBOLS AND ABBREVIATIONS:
    %
    % N:  Number of samples.
    %
    % q:  Number of outputs.
    %
    % nx:  Number of rule consequents.
    %
    % nz:  Number of rule premises.
    %
    % M: Number of local models.
    %
    %
    % EXPLANATIONS:
    %
    %   numberOfLMReliable:
    %   It decribes the number of local models that can guarantee a
    %   reliable model estimation.
    %
    %   idxAllowedLM:
    %   This logical vector stores the information if a local model is
    %   allowed to be splitted. That check is only needed if intertpretable
    %   areas are intended. If any areas are allowed, all possible splits
    %   can be done.
    %
    %   suggestedNet:
    %   It decribes which net structure in the net object has the best
    %   performance/complexity trade-off.
    %
    %   splits:
    %   This property describes how many splits should be taken along each
    %   dimension during the training.
    %
    %   outputModel:
    %   This is the output model of the net object. It is updated in each
    %   iteration. With each additional local model, the training model
    %   will generate a lower error.
    % 
    %   xRegressor/zRegressor:
    %   This properties store the regression matrix for the rule
    %   consequents and premises. This has a practical mean: Both only have
    %   to be calculated once during the training.
    %
    %   MSFValue:
    %   This property stores the Gaussians for each local model. The
    %   reason for this is similar to the regressor: Only the worst
    %   local model, and so its Gaussian, has to be splitted. The rest stay
    %   the same. Storing the rest saves computing time.
    %
    %
    % LoLiMoT - Nonlinear System Identification Toolbox
    % Torsten Fischer, 08-December-2011
    % Institute of Mechanics & Automatic Control, University of Siegen, Germany
    % Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles
    
   
    properties % properties that are not in the super-classes
        
        % Model parameters
        numberOfLMReliable = 0;             % 1 x 1           Number of LMs for a reliable model estimation
        
        % Constants
        splits = 1;                         % 1 x 1           Number of splits tested in each dimension
                
        % Data set definitions
        outputModel = [];                   % N x q           Model outputs
        xRegressor = [];                    % N x nx          Regression matrix/Regressors for rule consequents
        zRegressor = [];                    % N x nz          Regression matrix/Regressors for rule premises
        MSFValue = cell(1,1);               % 1 x M           Membership function values for each local model
        
    end
    
    methods % for the training algorithmn
        obj = train(obj,displayMode)
        obj = estimateFirstLM(obj)
        obj = estimateGivenModel(obj)
        obj = estimateModelGlobal(obj,iteration)
        [obj, forbiddenSplit] = LMSplitEstimate(obj, worstLM, splitDimension, splitRatio)
        worstLM = findWorstLM(obj)
        
    end
    
    methods(Static)
        [y name] = exampleFunctions(u,bsp)
        [lowerLeftCorner1, upperRightCorner1, lowerLeftCorner2, upperRightCorner2] = ...
            LMSplit(lowerLeftCorner, upperRightCorner, splitDimension, splitRatio)
    end
    
end

