function validityFunctionValue = calculateVFV(MSFValue)
% CALCULATEVFV calculates the validity function values of the leaf models
% of the local model object. Therefore it normalizes the membership
% functions of the leaf models. It is possible to hand a cell array
% or a matrix over. If the MSFValues are given as cells {1 x M}, a cell is
% returned {1 x M}. If a matrix is given (n x M), a matrix is returned (n x
% M)
%
% validityFunctionValue = calculateVFV(MSFValue)
%
%
% OUTPUTS:
%
% validityFunctionValue (1 x M) cell array or matrix of normalized
%                               membership function values
%                       
% INPUTS:
%
% MSFValue              (1 x M) cell array or matrix of (non-normalized)
%                               membership function values of the leaf
%                               models
%
%
% SYMBOLS AND ABBREVIATIONS:
%
% LM:  Local model
%
% p:   Number of inputs (physical inputs)
% q:   Number of outputs
% N:   Number of data samples
% M:   Number of LMs
% nx:  Number of regressors (x)
% nz:  Number of regressors (z)
%
%
% LMNtool - Local Model Network Toolbox
% Tobias Ebert, 03-November-2011
% Institute of Mechanics & Automatic Control, University of Siegen, Germany
% Copyright (c) 2012 by Prof. Dr.-Ing. Oliver Nelles

% 2011/11/18:   help updated
% 2011/11/03:   input of matrix is now possible


if iscell(MSFValue)
    
    % calculate sum
    sumMSFValue = sum(cell2mat(MSFValue),2);
    % normalize and superpose all active gaussians
    validityFunctionValue = cellfun(@(MSFOneCell) sum(bsxfun(@rdivide,MSFOneCell,sumMSFValue),2),MSFValue,'UniformOutput',false);
    
else
    
    % calculate sum
    sumMSFValue = sum(MSFValue,2);
    % normalize and superpose all active gaussians
    validityFunctionValue = bsxfun(@rdivide,MSFValue,sumMSFValue);
    
end

end
