function obj = functionGenerator(obj,inputDimension, numberOfTerms, meanExpoDist,type)
% FUNCTIONGENERATOR generates random polynomial functions.
%
% FUNCTIONGENERATOR generates polynomials of arbitrary <inputDimension>
% with few cross terms (i.e., u_1^pow_1*u_2^pow_2*...*u_p^pow_p). The
% number of polynomial terms is chosen with <numberOfTerms>. Their
% coefficients are chosen according to a normal distribution (mean 0,
% variance 1). Key is the generation of the powers pow_1,pow_2,...,pow_p
% for each input within each term. They are chosen according to a
% exponential distribution with mean <meanExpoDist>. For pow_i=0 the
% corresponding input is irrelevant for the considered term.
%
% INPUT:
% obj:              (1 x 1) - randomPolynomial object.
% inputDimension:   (1 x 1) - number of inputs.
% meanExpoDist:     (1 x 1)/(1 x d) - scalar or vector of means for the 
%                           exponential distribution to generation of the 
%                           powers. The vector entries speciefy the power
%                           for each dimension d.
% numberOfTerms:    (1 x 1)/(1 x d) - number of polynomial termsfor 
%                           all/each dimension. 
%
% OUTPUT:
% obj:              randomPolynomial object.
%
%
%   LMNtool - Local Model Network Toolbox
%   Torsten Fischer, 18-December-2014
%   Institute of Mechanics & Automatic Control, University of Siegen, Germany
%   Copyright (c) 2014 by Prof. Dr.-Ing. Oliver Nelles

% Check the input variables
if nargin < 3
    numberOfTerms = 10;
end
if nargin < 4
    meanExpoDist = 1.5;
end
if nargin < 5
    type = 'poly';
end

% Set type
obj.type = type;

% Calculate and store the (quasi-)random exponents
expo = 1; % expo > 1 makes pot values smaller. Values between 1 and 2 seem to be good.
% check if the complexity is equal in each dimension
if isscalar(numberOfTerms) && isscalar(meanExpoDist)
    obj.exponent = floor(exprnd(meanExpoDist,numberOfTerms,inputDimension).^(1/expo));
else
    % for a different complexity in each dimension, the number of terms can
    % be specified for each Dimension
    if isscalar(numberOfTerms)
        numberOfTerms=numberOfTerms*ones(1,inputDimension);
    end
    % the mean of the exponent can be specified for each dimension
    if isscalar(meanExpoDist)
        meanExpoDist=meanExpoDist*ones(1,inputDimension);
    end
    exponentMatrix = zeros(max(numberOfTerms),max(inputDimension));
    for ii=1:inputDimension
        exponentMatrix(1:numberOfTerms(ii),ii) = floor(exprnd(meanExpoDist(ii),numberOfTerms(ii),1).^(1/expo));
        exponentMatrix(:,ii)=exponentMatrix(randperm(max(numberOfTerms)),ii);
    end
    obj.exponent = exponentMatrix;
end

% Calculate and store random coefficients for the polynomial terms
obj.coefficients = randn(max(numberOfTerms),1);

    

% Calculate and store a random shift for each input.
% Restrict the possible input range for each axis between -1 and 1.
obj.shift = rand(inputDimension,1);

% Save 
obj.meanExpoDist = meanExpoDist;

end