function [output,terms] = calculateOutput(obj, input)
% CALCULATEPOLYNOMIALOUTPUT calculate the output of a randomPolynomial
% object.
%
% CALCULATEPOLYNOMIALOUTPUT calculate the output of a randomPolynomial
% object, using the given <input>. The properties of the object must be
% predefined.
%
% INPUT:
% obj:      object  randomPolynomial object
% input:    (N x p) input matrix
%
% OUTPUT:
% output:   (N x 1) output of the random polynomial, defined by the
%                   randomPolynomial object
%
%
%   SYMBOLS AND ABBREVIATIONS
%
%       p:   Number of inputs (physical inputs)
%       N:   Number of data samples
%
%
%   LMNtool - Local Model Network Toolbox
%   Torsten Fischer, 29-September-2014
%   Institute of Mechanics & Automatic Control, University of Siegen, Germany
%   Copyright (c) 2014 by Prof. Dr.-Ing. Oliver Nelles

% Check if there is a dimension missmatch
if size(obj.exponent,2) ~= size(input,2)
    error('randomPolynomial:calculatePolynomialOutput','Missmatch of the dimension of the given input and the dimension of the randomPolynomial object! Calculation aborded!')
end

% Check if the input range is correct
if any(max(input)>1) || any(min(input)<0)
    error('randomPolynomial:calculatePolynomialOutput','The input range is limited from 0 to 1.\nOtherwise the output values might become too large.\nTry another input matrix.')
end
% Get some constants
[numberOfSamples, numberOfInputs] = size(input);
numberOfTerms = size(obj.coefficients,1);

% Initialize the polynomial output
terms = ones(numberOfSamples,numberOfTerms);

% Calculate the polynomial terms.
for inp = 1:numberOfInputs
    terms = terms.*bsxfun(@power,input(:,inp)-obj.shift(inp),obj.exponent(:,inp)');
end

switch obj.type
    case 'poly'
        % Calculate the final output by multiplying the coefficients and summing up the terms
        output = terms*obj.coefficients;
    case 'sine'
        % Calculate the final output by multiplying the coefficients and summing up the terms
        output = sin(terms*pi)*obj.coefficients;
    case 'sigmoid'        
        % Calculate the sigmodial output
        output = sigmoid(10*(terms*obj.coefficients)./sqrt(numberOfTerms));
    otherwise
        error('randomPolynomial:calculateOutput','The given <type> is not implemented! Use one of the follwing: <poly>, <sinuns> or <sigmoid>.')
end

end

% function to calculate the sigmoid function
function y = sigmoid(x)
y = 1./(1+exp(-x));
end